<?php

namespace App\Http\Controllers;

use App\Models\Project;
use App\Models\ProjectCategory;
use App\Models\Country;
use App\Models\User;
use App\Models\TaskPhase;
use App\Models\Department;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;



class ProjectController extends Controller {
    public function index() {
        $projects = Project::with(['projectManager','salesPerson','department','country','projectCategory','projectSubCategory'])
            ->latest()
            ->paginate(10);
            //dd($projects);
        $mainCategories = ProjectCategory::with('subcategories')->whereNull('parent_id')->get();
       // dd($mainCategories);
        $countries = Country::all(); 
        $taskPhases = TaskPhase::all(); 
        $employees = User::all(); 
        $departments = Department::all(); 
    
        $projectManagers = User::whereHas('roles', function($query) {
            $query->where('name', 'Project Manager');
        })->get(); 
    
        $salesPersons = User::whereHas('roles', function ($query) {
            $query->where('name', 'Sales Team'); 
        })->get();
    
        return view('projects.index', compact(
            'projects', 'mainCategories', 'countries', 'projectManagers', 
            'taskPhases', 'employees', 'salesPersons', 'departments'
        ));
    }
    
    public function getSubcategories($parent_id) {
        $subcategories = ProjectCategory::where('parent_id', $parent_id)->get();
        return response()->json($subcategories);
    }    
    public function store(Request $request) {
        $validated = $request->validate([
            'name_or_url' => 'required|string|max:255',
            'dashboard_url' => 'nullable|url',
            'description' => 'nullable|string',
            'project_grade' => 'nullable|in:A,AA,AAA',
            'business_type' => 'nullable|in:Startup,Small,Mid-level,Enterprise',
            'project_category_id' => 'required|exists:project_categories,id',
            'project_subcategory_id' => 'nullable|exists:project_categories,id',
            'country_id' => 'required|exists:countries,id',
            'task_phases' => 'nullable|array',
            'project_manager_id' => 'required|exists:users,id',
            'assign_main_employee_id' => 'required|exists:users,id',
            'price' => 'nullable|numeric',
            'estimated_hours' => 'nullable|integer',
            'project_type' => 'nullable|in:Ongoing,One-time',
            'upwork_project_type' => 'nullable|in:Hourly,Fixed',
            'client_type' => 'nullable|in:New Client,Old Client',
            'report_type' => 'nullable|in:Weekly,Bi-Weekly,Monthly',
            'project_month' => 'nullable|date',
            'sales_person_id' => 'required|exists:users,id',
            'department_id' => 'required|exists:departments,id',
            'client_name' => 'nullable|string|max:255',
            'client_email' => 'nullable|email|max:255',
            'client_other_info' => 'nullable|string',
            'additional_employees' => 'nullable|array',

        ]);
        $validated['project_status'] = 'Working';

        $validated['created_by'] = Auth::id();


        Project::create($validated);
        return response()->json(['success' => 'Project created successfully.']);
    }
    public function edit(Project $project) {
        return response()->json($project);
    }

    public function update(Request $request, Project $project) {
        $validated = $request->validate([
            'name_or_url' => 'required|string|max:255',
            'dashboard_url' => 'nullable|url',
            'description' => 'nullable|string',
            'project_grade' => 'nullable|in:A,AA,AAA',
            'business_type' => 'nullable|in:Startup,Small,Mid-level,Enterprise',
            'project_category_id' => 'required|exists:project_categories,id',
            'project_subcategory_id' => 'nullable|exists:project_categories,id',
            'country_id' => 'required|exists:countries,id',
            'task_phases' => 'nullable|array',
            'project_manager_id' => 'required|exists:users,id',
            'assign_main_employee_id' => 'required|exists:users,id',
            'price' => 'nullable|numeric',
            'estimated_hours' => 'nullable|integer',
            'project_type' => 'nullable|in:Ongoing,One-time',
            'upwork_project_type' => 'nullable|in:Hourly,Fixed',
            'client_type' => 'nullable|in:New Client,Old Client',
            'report_type' => 'nullable|in:Weekly,Bi-Weekly,Monthly',
            'project_month' => 'nullable|date',
            'sales_person_id' => 'required|exists:users,id',
            'department_id' => 'required|exists:departments,id',
            'client_name' => 'nullable|string|max:255',
            'client_email' => 'nullable|email|max:255',
            'client_other_info' => 'nullable|string',
            'additional_employees' => 'nullable|array',
        ]);

        $project->update($validated);

        return response()->json(['success' => 'Project updated successfully.']);
    }
    public function destroy(Project $project) 
    {
        $project->delete();
        return response()->json(['success' => 'Project deleted successfully.']);
    }
    
    public function editStatus($id)
{
    $project = Project::findOrFail($id);
    return view('projects.status', compact('project'));
}

public function updateStatus(Request $request, $id)
{
    $request->validate([
        'project_status' => 'required',
        'status_date' => 'date',
        'reason_description' => 'nullable|string',
        'can_client_rehire' => 'nullable',
        'rehire_date' => 'nullable|date',
    ]);

    $project = Project::findOrFail($id);
    $project->update([
        'project_status' => $request->project_status,
        'status_date' => $request->status_date,
        'reason_description' => $request->reason_description,
        'can_client_rehire' => $request->can_client_rehire,
        'rehire_date' => $request->rehire_date,
    ]);

    return redirect()->route('projects.index')->with('success', 'Project status updated successfully.');
}
}


