<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Department;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Spatie\Permission\Traits\HasRoles; // Ensure this is in your User model
use Illuminate\Support\Facades\Log;


class UserController extends Controller
{
    public function index()
    {
        $users = User::with('roles','department','reportingPerson')->paginate(6);
        
      // dd($users);
        $allUsersForDropdown = User::all(); // Fetch all users for the dropdown

       $departments = Department::all();
       
        return view('users.index', compact('users','departments','allUsersForDropdown'));
    }

    public function generateEmployeeCode()
    {
        $latestUser = User::latest()->first();
        $nextNumber = $latestUser ? ((int) substr($latestUser->employee_code, -3)) + 1 : 1;
        $generatedCode = 'SEO-' . str_pad($nextNumber, 3, '0', STR_PAD_LEFT);
    
        return response()->json(['code' => $generatedCode]);
    }
    


    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'phone_number' => 'nullable|string|max:20',
            //'monthly_target' => 'nullable|numeric|min:0',
            //'upsell_incentive' => 'nullable|numeric|min:0|max:100',
           // 'department' => 'nullable|exists:departments,id',
           // 'disable_login' => 'nullable|boolean',
            //'password' => 'nullable|min:6|confirmed',
            'employee_code' => 'required|string|max:50|unique:users,employee_code',
            'employee_code' => $request->role !== 'Admin' ? 'required|string|max:20|unique:users,employee_code' : 'nullable',
            'image' => 'nullable|image|mimes:jpg,png,jpeg,gif|max:2048',
            'role' => 'required|string|exists:roles,name', // Ensure role exists in Spatie's roles table

        ]);
    
        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors(),
            ], 422);
        }
     // Generate employee code if not provided
     if (!$request->filled('employee_code')) {
        $latestUser = User::latest()->first();
        $nextNumber = $latestUser ? ((int) substr($latestUser->employee_code, -3)) + 1 : 1;
        $generatedCode = 'EMP-' . str_pad($nextNumber, 3, '0', STR_PAD_LEFT);
    } else {
        $generatedCode = $request->employee_code;
    }
        // Remove commas from 'monthly_target' before saving
        $monthlyTarget = $request->monthly_target ? str_replace(',', '', $request->monthly_target) : null;
    
        $imagePath = $request->file('image') 
            ? $request->file('image')->store('images', 'public') 
            : null;
    
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'phone_number' => $request->phone_number,
            'monthly_target' => $monthlyTarget,
            'upsell_incentive' => $request->upsell_incentive,
            'department_id' => $request->department,
            'reporting_person' => $request->reporting_person,
            'allow_all_projects' => $request->boolean('allow_view_all_projects'),
            'disable_login' => $request->boolean('disable_login'),
            'experience' => $request->experience,
            'qualification' => $request->qualification,
            'specialization' => $request->specialization,
            'date_of_joining' => $request->date_of_joining,
            'password' => $request->password ? Hash::make($request->password) : null,
           'employee_code' => $generatedCode,
            'image' => $imagePath,
        ]);
    
        if ($request->role) {
            $user->assignRole($request->role);
        }
    
        return response()->json(['success' => 'User created successfully']);
    }
    
    public function edit(User $user)
    {
        return response()->json([
            'id' => $user->id,
            'name' => $user->name,
            'email' => $user->email,
            'phone_number' => $user->phone_number,
            'monthly_target' => $user->monthly_target,
            'upsell_incentive' => $user->upsell_incentive,
            'reporting_person' => $user->reporting_person,
            'department' => $user->department_id, // Ensure ID is returned, not object
            'allow_view_all_projects' => $user->allow_all_projects,
            'disable_login' => $user->disable_login,
            'experience' => $user->experience,
            'qualification' => $user->qualification,
            'specialization' => $user->specialization,
            'date_of_joining' => $user->date_of_joining,
            'employee_code' => $user->employee_code,
            'image' => $user->image ? asset('storage/' . $user->image) : null,
            'role' => $user->roles->pluck('name')->first(), // Get the first assigned role
        ]);
    }
    
    
    public function update(Request $request, User $user)
    {
        Log::info('Update function called for user ID: ' . $user->id);
    
        $validator = Validator::make($request->all(), [
            'email' => "nullable|email|unique:users,email,{$user->id},id",
            'phone_number' => 'nullable|string|max:20',
            'image' => 'nullable|image|mimes:jpg,png,jpeg,gif|max:2048',
            'role' => 'nullable|string|exists:roles,name', 
            'password' => 'nullable|min:6',
            'employee_code' => "nullable|string|max:50|unique:users,employee_code,{$user->id},id"
        ]);
    
        if ($validator->fails()) {
            Log::error('Validation failed:', $validator->errors()->toArray());
            return response()->json(['errors' => $validator->errors()], 422);
        }
    
        // Handle image upload
        if ($request->hasFile('image')) {
            if ($user->image) {
                Storage::disk('public')->delete($user->image);
            }
            $imagePath = $request->file('image')->store('images', 'public');
        } else {
            $imagePath = $user->image;
        }
    
        $updateData = [
            'name' => $request->name ?? $user->name,
            'email' => $request->email ?? $user->email,
            'phone_number' => $request->phone_number ?? $user->phone_number,
            'monthly_target' => $request->monthly_target ? str_replace(',', '', $request->monthly_target) : $user->monthly_target,
            'upsell_incentive' => $request->upsell_incentive ?? $user->upsell_incentive,
            'department_id' => $request->department ?? $user->department_id,
            'allow_all_projects' => $request->has('allow_view_all_projects') ? $request->boolean('allow_view_all_projects') : $user->allow_all_projects,
            'disable_login' => $request->disable_login ?? $user->disable_login,
            'experience' => $request->experience ?? $user->experience,
            'reporting_person' => $request->reporting_person ?? $user->reporting_person,
            'qualification' => $request->qualification ?? $user->qualification,
            'specialization' => $request->specialization ?? $user->specialization,
            'date_of_joining' => $request->date_of_joining ?? $user->date_of_joining,
            'image' => $imagePath,
            'password' => $request->filled('password') ? Hash::make($request->password) : $user->password,
        ];
    
        if ($request->filled('employee_code')) {
            $updateData['employee_code'] = $request->employee_code;
        }
    
        Log::info('Update data:', $updateData);  // Log the update data before updating
    
        // Update user
        $updated = $user->update($updateData);
        
        if (!$updated) {
            Log::error('User update failed for ID: ' . $user->id);
            return response()->json(['error' => 'Failed to update user'], 500);
        }
    
        Log::info('User updated successfully for ID: ' . $user->id);
    
        // Handle Role Update with Spatie
        if ($request->filled('role')) {
            $user->syncRoles([$request->role]); 
            Log::info('Role updated successfully');
        }
    
        return response()->json(['success' => 'User updated successfully']);
    }
    
    

    public function destroy(User $user)
    {
        if ($user->image) {
            Storage::delete($user->image);
        }
        $user->delete();

        return response()->json(['success' => 'User deleted successfully']);
    }
    public function filterUsers(Request $request)
    {
        $usersQuery = User::query();
    
        // Filter by role
        if ($request->filled('role')) {
            $usersQuery->whereHas('roles', function ($query) use ($request) {
                $query->where('name', $request->role);
            });
        }
    
        // Filter by department
        if ($request->filled('department')) {
            $usersQuery->where('department_id', $request->department);
        }
    
        // Filter by reporting person
        if ($request->filled('reporting_person')) {
            $usersQuery->where('reporting_person', $request->reporting_person);
        }
    
        // Count total filtered users
        $totalUsers = $usersQuery->count();
    
        // Fetch users with pagination if more than 6 exist, else get all
        $users = $totalUsers > 6 ? $usersQuery->paginate(6) : $usersQuery->get();
    
        if ($request->ajax()) {
            return response()->json([
                'html'       => view('users.users_rows', compact('users'))->render(),
                'pagination' => $totalUsers > 6 ? $users->links()->toHtml() : '',
            ]);
        }
    
        return view('users.index', compact('users'));
    }
    

    
    
    
    public function show($id)
    {
        return response()->json(['message' => 'Show method not implemented'], 404);
    }
     

}
