<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Candidate;
use App\Models\User;

use App\Models\Department;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\UserController;



class CandidateController extends Controller
{
    public function index(Request $request)
    {
        $query = Candidate::query();
    
        // Apply filters dynamically if present
        if (!empty($request->name)) {
            $query->where('name', 'like', '%' . $request->name . '%');
        }
    
        if (!empty($request->filter_department)) {  // Fixed incorrect key
            $query->where('department_id', $request->filter_department);
        }
    
        if (!empty($request->filter_status)) {  // Fixed incorrect key
            $query->where('status', $request->filter_status);
        }
    
        if (!empty($request->filter_added_by)) {  // Fixed incorrect key
            $query->where('added_by', $request->filter_added_by);
        }
    
        // Fetch candidates with related department and user info for better performance
        $candidates = $query->with(['department', 'addedBy'])->paginate(10);
    
        // Fetch all departments
        $departments = Department::all();
    
        // Fetch users who have HR or Admin roles
        $users = User::whereHas('roles', function ($q) {
            $q->whereIn('name', ['HR', 'Admin']);
        })->get();
    
        return view('candidates.index', compact('candidates', 'departments', 'users'));
    }
    
    

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:candidates,email',
            'phone_number' => 'required|string',
            'experience' => 'nullable|string',
            'current_salary' => 'nullable|string',
            'expected_salary' => 'nullable|string',
            'offered_salary' => 'nullable|string',
            'date_of_joining' => 'nullable|date',
            'comments' => 'nullable|string',
            'resume' => 'nullable|file|mimes:pdf,doc,docx',
            'department_id' => 'required|exists:departments,id',
            'status' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }
        

        $data = $request->except(['resume']);
        if ($request->hasFile('resume')) {
            $data['resume'] = $request->file('resume')->store('resumes', 'public');
        }
          // Store the ID of the authenticated user who added the candidate
        $data['added_by'] = auth()->id();
    


        Candidate::create($data);

        return response()->json(['message' => 'Candidate added successfully!']);
    }

    public function edit($id)
    {
        $candidate = Candidate::findOrFail($id);
        return response()->json($candidate);
    }

    public function update(Request $request, $id)
    {
        $candidate = Candidate::findOrFail($id);
    
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:candidates,email,' . $id,
            'phone_number' => 'required|string',
            'experience' => 'nullable',
            'current_salary' => 'nullable',
            'expected_salary' => 'nullable',
            'offered_salary' => 'nullable',
            'date_of_joining' => 'nullable|date',
            'department_id' => 'nullable|exists:departments,id',
            'status' => 'required|string',
            'comments' => 'nullable|string',
            'resume' => 'nullable|file|mimes:pdf|max:2048',
        ]);
    
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }
    
        $data = $request->except(['resume']);
    
        if ($request->hasFile('resume')) {
            $data['resume'] = $request->file('resume')->store('resumes', 'public');
        }
    
        $updated = $candidate->update($data);
    
        if ($updated) {
            return response()->json(['message' => 'Candidate updated successfully!']);
        } else {
            return response()->json(['error' => 'Failed to update candidate'], 500);
        }
    }

    public function destroy(Candidate $candidate)
{
    if ($candidate->resume) {
        Storage::delete($candidate->resume); // Deletes the stored file
    }

    $candidate->delete();

    return response()->json(['success' => 'Candidate deleted successfully']);
}
public function show(Candidate $candidate)
{
    return response()->json($candidate);
}

 
}    